/**
 * @~japanese
 * @file openEL.h
 * @brief		OpenEL 共通ヘッダーファイル
 * @Version 3.0.0
 *
 * @~english
 * @file openEL.h
 * @brief		OpenEL Common Header File
 * @Version 3.0.0
 */
/*

Copyright (c) 2017,2018 Japan Embedded Systems Technology Association(JASA)
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    Neither the name of the Association nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef OPENEL_H
#define OPENEL_H

#ifdef __cplusplus
namespace el {
extern "C" {
#endif /* __cplusplus */

#include <stdint.h>

/* Macro (global scope) */
/*---------------------------------------------------------------------------*/

#define OPENEL_MAJOR      3
#define OPENEL_MINOR      0
#define OPENEL_PATCHLEVEL 0
#define OPENEL_VERSION "OpenEL 3.0.0"

/* C PSM */
enum ReturnCode {
  HAL_OK = 0,
	HAL_ERROR
};

/* typedef definition */
typedef float  float32_t;
typedef double float64_t;
#if 1
  typedef float32_t halfloat;
#else
  typedef float64_t halfloat;
#endif

typedef struct HALId {
  int32_t deviceKindId;
	int32_t vendorId;
	int32_t productId;
	int32_t instanceId;
} HALID_T;

typedef struct Property {
	HALID_T halId;
	char deviceName[32];
	char functionList[32][32];
} PROPERTY_T;

typedef union HalComponent HALCOMPONENT_T;

typedef struct HALObserver {
  void (*notify_event)(HALCOMPONENT_T *halComponent, int32_t event_id);
	void (*notify_error)(HALCOMPONENT_T *halComponent, int32_t error_id);
} HALOBSERVER_T;

typedef struct HALObserver HALOBSERVER_T;

#define HALCOMPONENT_MEMBER \
  int32_t handle; \
  HALID_T halId;\
  PROPERTY_T property;\
  HALOBSERVER_T observerList[8];\
  int32_t time;

typedef struct HALComponent {
  HALCOMPONENT_MEMBER
} HALCOMPONENT_CORE_T;

enum ReturnCode HalInit(HALCOMPONENT_T *halComponent);
enum ReturnCode HalReInit(HALCOMPONENT_T *halComponent);
enum ReturnCode HalFinalize(HALCOMPONENT_T *halComponent);
enum ReturnCode HalAddObserver(HALCOMPONENT_T *halComponent, HALOBSERVER_T *halObserver);
enum ReturnCode HalRemoveObserver(HALCOMPONENT_T *halComponent, HALOBSERVER_T *halObserver);
enum ReturnCode HalGetProperty(HALCOMPONENT_T *halComponent, PROPERTY_T *property);
enum ReturnCode HalGetTime(HALCOMPONENT_T *halComponent, int32_t *time);

typedef struct EventTimer EVENTTIMER_T;

typedef struct TimerObserver {
  void (*notify_timer)(EVENTTIMER_T *eventTimer);
} TIMEROBSERVER_T;

typedef struct EventTimer {
  TIMEROBSERVER_T observerList[8];
} EVENTTIMER_T;

enum ReturnCode HalEventTimerStartTimer(EVENTTIMER_T *eventTimer);
enum ReturnCode HalEventTimerStopTimer(EVENTTIMER_T *eventTimer);
enum ReturnCode HalEventTimerSetEventPeriod(EVENTTIMER_T *eventTimer, int32_t eventPeriod);
enum ReturnCode HalEventTimerAddObserver(EVENTTIMER_T *eventTimer, TIMEROBSERVER_T *timerObserver);
enum ReturnCode HalEventTimerRemoveObserver(EVENTTIMER_T *eventTimer, TIMEROBSERVER_T *timerObserver);


typedef struct PositionControlActuator {
  /* HALCOMPONENT */
  HALCOMPONENT_MEMBER
  /* POSITIONCONTROLACTUATOR */
	halfloat actualPosition;
} POSITIONCONTROLACTUATOR_T;

enum ReturnCode HalSetPosition(HALCOMPONENT_T *positionControlActuator, halfloat position);
enum ReturnCode HalGetPosition(HALCOMPONENT_T *positionControlActuator, halfloat *position);

typedef struct VelocityControlActuator {
  /* HALCOMPONENT */
  HALCOMPONENT_MEMBER
  /* VELOCITYCONTROLACTUATOR */
  halfloat actualVelocity;
} VELOCITYCONTROLACTUATOR_T;

enum ReturnCode HalSetVelocity(HALCOMPONENT_T *velocityControlActuator, halfloat velocity);
enum ReturnCode HalGetVelocity(HALCOMPONENT_T *velocityControlActuator, halfloat *velocity);

typedef struct TorqueControlActuator {
  /* HALCOMPONENT */
  HALCOMPONENT_MEMBER
  /* TORQUECONTROLACTUATOR */
  halfloat actualTorque;
} TORQUECONTROLACTUATOR_T;

enum ReturnCode HalSetTorque(HALCOMPONENT_T *torqueControlActuator, halfloat torque);
enum ReturnCode HalGetTorque(HALCOMPONENT_T *torqueControlActuator, halfloat *torque);

typedef struct TorqueSensor {
  /* HALCOMPONENT */
  HALCOMPONENT_MEMBER
	/* SENSOR */
	halfloat value[32];
	char units[32][32];
} TORQUESENSOR_T;

enum ReturnCode HalTorqueSensorGetValue(HALCOMPONENT_T *torqueSensor, halfloat *list);
enum ReturnCode HalTorqueSensorGetTimedValue(HALCOMPONENT_T *torqueSensor, halfloat *list, int32_t *timer);
enum ReturnCode HalTorqueSensorSetUnit(HALCOMPONENT_T *torqueSensor, char units);
enum ReturnCode HalTorqueSensorGetUnit(HALCOMPONENT_T *torqueSensor, char *units);
enum ReturnCode HalTorqueSensorSetReset(HALCOMPONENT_T *torqueSensor);
enum ReturnCode HalTorqueSensorStart(HALCOMPONENT_T *torqueSensor);
enum ReturnCode HalTorqueSensorStop(HALCOMPONENT_T *torqueSensor);

typedef union HalComponent {
  HALCOMPONENT_CORE_T halComponent_core;
  POSITIONCONTROLACTUATOR_T positionControlActuator;
  VELOCITYCONTROLACTUATOR_T velocityControlActuator;
  TORQUECONTROLACTUATOR_T torqueControlActuator;
  TORQUESENSOR_T torqueSensor;
} HALCOMPONENT_T;

/* The end of C PSM */

enum ReturnCode Dummy();

/** general component --- function table */
typedef struct elGenaralFncTblSt {
	/* 0x00 */ enum ReturnCode (*pFncInit)(); /**< Initialize */
	/* 0x01 */ enum ReturnCode (*pFncReInit)(); /**< ReInit */
	/* 0x02 */ enum ReturnCode (*pFncFinalize)(); /**< Finalize */
	/* 0x03 */ enum ReturnCode (*pFncAddObserver)(); /**< AddObserver */
	/* 0x04 */ enum ReturnCode (*pFncRemoveObserver)(); /**< RemoveObserver */
	/* 0x05 */ enum ReturnCode (*pFncGetProperty)(); /**< GetProperty */
	/* 0x06 */ enum ReturnCode (*pFncGetTime)(); /**< GetTime */
	/* 0x07 */ enum ReturnCode (*pFncDummy07)(); /**< Reserved */

  /* 0x08 */ enum ReturnCode (*pFncDummy08)(); /**< Reserved */
	/* 0x09 */ enum ReturnCode (*pFncDummy09)(); /**< Reserved */
  /* 0x0A */ enum ReturnCode (*pFncDummy0A)(); /**< Reserved */
	/* 0x0B */ enum ReturnCode (*pFncDummy0B)(); /**< Reserved */
  /* 0x0C */ enum ReturnCode (*pFncDummy0C)(); /**< Reserved */
	/* 0x0D */ enum ReturnCode (*pFncDummy0D)(); /**< Reserved */
	/* 0x0E */ enum ReturnCode (*pFncDummy0E)(); /**< Reserved */
	/* 0x0F */ enum ReturnCode (*pFncDummy0F)(); /**< Reserved */

	/* 0x10 */ enum ReturnCode (*pFncDummy10)(); /**< Reserved */
	/* 0x11 */ enum ReturnCode (*pFncDummy11)(); /**< Reserved */
	/* 0x12 */ enum ReturnCode (*pFncDummy12)(); /**< Reserved */
	/* 0x13 */ enum ReturnCode (*pFncDummy13)(); /**< Reserved */
	/* 0x14 */ enum ReturnCode (*pFncDummy14)(); /**< Reserved */
	/* 0x15 */ enum ReturnCode (*pFncDummy15)(); /**< Reserved */
	/* 0x16 */ enum ReturnCode (*pFncDummy16)(); /**< Reserved */
	/* 0x17 */ enum ReturnCode (*pFncDummy17)(); /**< Reserved */

	/* 0x18 */ enum ReturnCode (*pFncDummy18)(); /**< Reserved */
	/* 0x19 */ enum ReturnCode (*pFncDummy19)(); /**< Reserved */
	/* 0x1A */ enum ReturnCode (*pFncDummy1A)(); /**< Reserved */
	/* 0x1B */ enum ReturnCode (*pFncDummy1B)(); /**< Reserved */
	/* 0x1C */ enum ReturnCode (*pFncDummy1C)(); /**< Reserved */
	/* 0x1D */ enum ReturnCode (*pFncDummy1D)(); /**< Reserved */
	/* 0x1E */ enum ReturnCode (*pFncDummy1E)(); /**< Reserved */
	/* 0x1F */ enum ReturnCode (*pFncDummy1F)(); /**< Reserved */
} EL_GENERAL_FNC_TBL_T;

/** function table universal */
typedef union elCommonFncTbl {
	EL_GENERAL_FNC_TBL_T *pGeneral;
	#define EL_SW_COMPOMENT_PROPERTY_TABLE 1
	#include "openEL_profCnf.h"
	#undef EL_SW_COMPOMENT_PROPERTY_TABLE
} EL_CMN_FNC_TBL_T;
extern const uint32_t elSzCompoTbl;

/* Variable (global scope) */
/*---------------------------------------------------------------------------*/
struct elPort_st {
	uint32_t physicalPortID;
	EL_CMN_FNC_TBL_T fncTbl;
	uint32_t deviceID;
};

struct elCompoTbl_st {
	uint32_t vendorID; /* vendor */
	uint32_t componentID; /* component */
	EL_CMN_FNC_TBL_T *pFncTbl;
};
extern const struct elCompoTbl_st elCompoTbl[];

#define EL_PORT_ID_MAX (16) /**<  */

/* Variable (friend scope) */
/*---------------------------------------------------------------------------*/
#if 1==OPENEL_SW_SURFACE_FRIEND
extern struct elPort_st elPortTbl[EL_PORT_ID_MAX];
#endif /* OPENEL_SW_SURFACE_FRIEND */

/*---------------------------------------------------------------------------*/
/** 物理portIDに変換 */
int32_t elGetPhyicalPortID(uint32_t portID,uint32_t PropertyID);
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* openEL component Property Integrate */
#define EL_SW_COMPONENT_PROPERTY_HEADER 1
#include "openEL_profCnf.h"
/*---------------------------------------------------------------------------*/

#ifdef __cplusplus
} /* extern "C" */
} /* namespace el */
#endif /* __cplusplus */

#endif /* OPENEL_H */
