/**
 * @~japanese
 * @file openEL_motor.h
 * @brief		OpenEL モーター共有部
 * @Version 3.0.0
 *
 * @~english
 * @file openEL_motor.h
 * @brief		OpenEL Motor Common File
 * @Version 3.0.0
 */
/*

Copyright (c) 2017,2018 Japan Embedded Systems Technology Association(JASA)
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    Neither the name of the Association nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef OPENEL_MOTOR_H_
#define OPENEL_MOTOR_H_

#include "openEL.h"

#define ELMOTOR_PROPERTY (0x0010)

#if 0
//#define EL_NXT_PHYSICAL_ID_MOTOR     (0x00100000)

/* Control Mode */
#define ELMOTOR_CTL_NON			(0) /**< Control Disable */
#define ELMOTOR_CTL_TORQUE		(1) /**< Torque Control Enable */
#define ELMOTOR_CTL_SPEED		(2) /**< Speed Control Enable */
#define ELMOTOR_CTL_POSITION	(3) /**< Position Control Enable */


//i 後で整理
/* Return Value */
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
#define ELRETURN_OK				(0)
#define ELRETURN_ERR_NC			(-256 + 1)	/**< 未接続 */
#define ELRETURN_ERR_PORT		(-256 + 2)	/**< ポート未接続 */
#define ELRETURN_ERR_PRM		(-256 + 3)	/**< パラメーター番号未対応 */
#define ELRETURN_ERR_NM			(-256 + 4)	/**< 未実装 */



/* Parameter ID */
#define ELMOTOR_PRM_SW_INT_FL	(8)  	/**< 整数 / 不動小数点 */
#define ELMOTOR_PRM_SW_INC_ABS	(16)	/**< 0:Incremental / 1:Absolute */
#define ELMOTOR_PRM_RESO_BASE	(18)	/**< 基本分解能 */
#define ELMOTOR_PRM_N_MUL_GEAR	(20)	/**< 電子ギア 分母 */
#define ELMOTOR_PRM_N_DIV_GEAR	(22)	/**< 電子ギア 分母 */
#define ELMOTOR_PRM_ONE_TURN	(24)	/**< 設定単位/rev */
#define ELMOTOR_PRM_K_MOVE		(28)	/**< 移動量換算定数 */
#define ELMOTOR_PRM_TORQUE_MAX_I32	(32)	/**< 最大トルク  整数 */
#define ELMOTOR_PRM_TORQUE_MAX_F64 	(36)	/**< 最大トルク  浮動小数点 */
#define ELMOTOR_PRM_TORQUE_LIM  	(40)	/**< トルク制限 */
#define ELMOTOR_PRM_TORQUE_LIM_STOP (41)	/**< トルク制限 停止時 */

/* callback event ID */
#define ELMOTOR_EVENT_ARRIVE_POSITION	(0)
#define ELMOTOR_EVENT_PASSAGE_POSITION	(1)
#define ELMOTOR_EVENT_ARRIVE_SPEED		(2)

/**
 * @~japanese 信号入力
 * @~english signal input
 */
typedef struct elMotorSigIn_st {
	uint32_t _not_use0:8;	/**< @~japanese 未使用 0:固定 (将来の拡張性のため 0 にする)  @~english reserve */
	uint32_t _not_use1:8;	/**< @~japanese 未使用 0:固定 (将来の拡張性のため 0 にする)  @~english reserve */
	uint32_t _vender_specific:16;	/**< @~japanese  ベンダー個別仕様  @~english Vendor specific  */
} ELMOTOR_SIGIN_T;

/**
 * @~japanese 信号出力
 * @~english signal output
 */
typedef struct elMotorSigOut_st {
	uint32_t alarm:1;		/**< アラーム 0:なし / 1:発生 */
	uint32_t warning:1;		/**< ワーニング 0:なし / 1:発生 */
	uint32_t _not_use_0:6; 	/**< 未使用 0:固定 (将来の拡張性のため 0 にする) */

	uint32_t inPos:1;		/**< 位置制御 位置決め完了 */
	uint32_t inVel:1;		/**< 速度制御 目標速度到達 */
	uint32_t trqP:1;		/**< トルク 正転側 制限値到達 */
	uint32_t trqN:1;		/**< トルク 逆転側 制限値到達 */
	uint32_t _not_use_1:4;	/**< 未使用 0:固定 (将来の拡張性のため 0 にする) */

	uint32_t _vender_specific:16;	/**< ベンダー個別仕様 */
} ELMOTOR_SIGOUT_T;

/**
 * @~japanese コールバック関数の形式宣言
 * @~english form of callback function
 */
typedef void (*ELMOTOR_P_FNC_CALLBACK)(uint32_t portID,uint32_t eventID); /**<  */

#endif

/**
 * @~japanese コンポーネント 関数テーブル
 * @~english component function table
 */
typedef struct elMotorFncTblSt {
	/* 0x00 */ enum ReturnCode (*pFncInit)(); /**< Initialize */
	/* 0x01 */ enum ReturnCode (*pFncReInit)(); /**< ReInit */
	/* 0x02 */ enum ReturnCode (*pFncFinalize)(); /**< Finalize */
	/* 0x03 */ enum ReturnCode (*pFncAddObserver)(); /**< AddObserver */
	/* 0x04 */ enum ReturnCode (*pFncRemoveObserver)(); /**< RemoveObserver */
	/* 0x05 */ enum ReturnCode (*pFncGetProperty)(); /**< GetProperty */
	/* 0x06 */ enum ReturnCode (*pFncGetTime)(); /**< GetTime */
	/* 0x07 */ enum ReturnCode (*pFncDummy07)(); /**< Reserved */

  /* 0x08 */ enum ReturnCode (*pFncSetPosition)(HALCOMPONENT_T **halComponent, halfloat position); /**< SetPosition */
	/* 0x09 */ enum ReturnCode (*pFncGetPosition)(HALCOMPONENT_T **halComponent, halfloat **position); /**< GetPosition */
	/* 0x0A */ enum ReturnCode (*pFncDummy0A)(); /**< Reserved */
	/* 0x0B */ enum ReturnCode (*pFncDummy0B)(); /**< Reserved */
  /* 0x0C */ enum ReturnCode (*pFncDummy0C)(); /**< Reserved */
	/* 0x0D */ enum ReturnCode (*pFncDummy0D)(); /**< Reserved */
	/* 0x0E */ enum ReturnCode (*pFncDummy0E)(); /**< Reserved */
	/* 0x0F */ enum ReturnCode (*pFncDummy0F)(); /**< Reserved */

	/* 0x10 */ enum ReturnCode (*pFncDummy10)(); /**< Reserved */
	/* 0x11 */ enum ReturnCode (*pFncDummy11)(); /**< Reserved */
	/* 0x12 */ enum ReturnCode (*pFncDummy12)(); /**< Reserved */
	/* 0x13 */ enum ReturnCode (*pFncDummy13)(); /**< Reserved */
	/* 0x14 */ enum ReturnCode (*pFncDummy14)(); /**< Reserved */
	/* 0x15 */ enum ReturnCode (*pFncDummy15)(); /**< Reserved */
	/* 0x16 */ enum ReturnCode (*pFncDummy16)(); /**< Reserved */
	/* 0x17 */ enum ReturnCode (*pFncDummy17)(); /**< Reserved */

	/* 0x18 */ enum ReturnCode (*pFncDummy18)(); /**< Reserved */
	/* 0x19 */ enum ReturnCode (*pFncDummy19)(); /**< Reserved */
	/* 0x1A */ enum ReturnCode (*pFncDummy1A)(); /**< Reserved */
	/* 0x1B */ enum ReturnCode (*pFncDummy1B)(); /**< Reserved */
	/* 0x1C */ enum ReturnCode (*pFncDummy1C)(); /**< Reserved */
	/* 0x1D */ enum ReturnCode (*pFncDummy1D)(); /**< Reserved */
	/* 0x1E */ enum ReturnCode (*pFncDummy1E)(); /**< Reserved */
	/* 0x1F */ enum ReturnCode (*pFncDummy1F)(); /**< Reserved */
} ELMOTOR_FNC_TBL_T;

#if 0
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * モーター制御モードの設定
 * @param[in] portID 論理ポートID
 * @param[in] mode 位置決め制御 0:無効 / 1:位置制御 / 2:速度制御 / 3:トルク制御
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Sets the motor control mode
 * @param[in] portID logical port ID
 * @param[in] mode 0:disable / 1:position control / 2:speed control / 3:torque control
 * @return the error code  0:OK / Not 0: error message code
 */
int32_t elMotorSetCtlMode(uint32_t portID,uint32_t mode);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * モーター制御状態の取得
 * @param[in] portID 論理ポートID
 * @param[out] *pMode モーター制御モード
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Gets the motor control mode
 * @param[in] portID logical port ID
 * @param[in] *pMode motor control mode
 * @return The error Code  0:OK / Not 0: error message code
 */
int32_t elMotorGetCtlMode(uint32_t portID,uint32_t *pMode);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * 指定位置への位置決め運転 整数版
 * @param[in] portID 論理ポートID
 * @param[in] position 目標位置指令 単位[Step]
 * @param[in] speed 定常速度 単位[Hz]
 * @param[in] tmAcc 定常速度までの加速時間 単位[ms]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Operates to index position , integer type
 * @param[in] portID logical port ID
 * @param[in] position target position. unit[Step]
 * @param[in] speed steady speed. unit[Hz]
 * @param[in] tmAcc accerating time until steady speed. unit[ms]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorSetPosition_I32(uint32_t portID,int32_t position,int32_t speed,int32_t tmAcc);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * 指定位置への位置決め運転 , 浮動小数点型
 * @param[in] portID 論理ポートID
 * @param[in] position 目標位置指令 単位[radian] または [m]
 * @param[in] speed 定常速度 単位[radian/s] または [m/s]
 * @param[in] tmAcc 定常速度までの加速時間 単位[s]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Moves to target position , floating point type
 * @param[in] portID logical port ID
 * @param[in] position target position. unit[radian] or [m]
 * @param[in] speed steady speed. unit [radian/s] or [m/s]
 * @param[in] tmAcc accelarating time until steady speed. unit[s]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorSetPosition_F64(uint32_t portID,double position,double speed,double tmAcc);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * 指令速度での運転
 * @param[in] portID 論理ポートID
 * @param[in] speed 目標速度 単位[Hz]
 * @param[in] tmAcc 目標速度までの加速時間 単位[ms]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Operates
 * @param[in] portID logical port ID
 * @param[in] target spped. unit[Hz]
 * @param[in] tmAcc accelarating time until target spped. unit[ms]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorSetSpeed_I32(uint32_t portID,int32_t speed,int32_t tmAcc);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * 指令速度での運転
 * @param[in] portID 論理ポートID
 * @param[in] speed 目標速度 単位[radian/s] または [m/s]
 * @param[in] tmAcc 目標速度までの加速時間 単位[s]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Operates
 * @param[in] portID logical port ID
 * @param[in] target spped. unit[radian/s] or [m/s]
 * @param[in] tmAcc accelarating time until target spped. unit[ms]
 * @return the error Code  0:OK / Not 0: error message code
 */
/*---------------------------------------------------------------------------*/
int32_t elMotorSetSpeed_F64( uint32_t portID,double speed,double tmAcc);
/**
 * @~japanese
 * トルク指令の設定 (トルク制御モード)
 * @param[in] portID 論理ポートID
 * @param[in] torque トルク指令 単位 0.1[%]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * sets commanded torque (for torque control mode)
 * @param[in] portID logical port ID
 * @param[in] torque commanded torque
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorSetTorque_I32(uint32_t portID,int32_t torque);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * トルク指令の設定 (トルク制御モード)
 * @param[in] portID 論理ポートID
 * @param[in] torque トルク指令 単位[Nm] or [N]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * sets commanded torque (for torque control mode)
 * @param[in] portID logical port ID
 * @param[in] torque commanded torque. unit[Nm] or [N]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorSetTorque_F64(uint32_t portID,double torque);
/*---------------------------------------------------------------------------*/
/* メモ
 * 仕様書(Ver.1.0用)とヘッダーファイルが違っている
 * 仕様書(Ver.1.0用)はVer.0.1で記載されています
 */
/**
 * @~japanese
 * 停止
 * @param[in] portID 論理ポートID
 * @param[in] ctlAft 停止後の制御モード 0:無効 / 1:続行
 * @param[in] tmDec 減速時間 単位[ms]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Brake
 * @param[in] portID logical port ID
 * @param[in] ctlAft control mode after stopping 0:disable / 1:continue
 * @param[in] tmDec decelaration time. unit[ms]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorSetBrake_I32(uint32_t portID,uint32_t ctlAftBrk,int32_t tmDec);
/**
 * @~japanese
 * 停止
 * @param[in] portID 論理ポートID
 * @param[in] ctlAft 停止後の制御モード 0:無効 / 1:続行
 * @param[in] tmDec 減速時間 単位[s]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * Brake
 * @param[in] portID logical port ID
 * @param[in] ctlAft control mode after stopping 0:disable / 1:continue
 * @param[in] tmDec decelaration time. unit[s]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorSetBrake_F64(uint32_t portID,uint32_t ctlAftBrk,double tmDec);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * 停止制御状態の取得
 * @param[in] portID 論理ポートID
 * @param[out] *pFlagBrake 停止ステータス 1:制動中 / 0:停止制御に入っていない
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * gets stpping control status
 * @param[in] portID logical port ID
 * @param[out] *pFlagBrake stopping status 1:stopping control / 0:expect stpping control
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorGetBrake(uint32_t portID,uint32_t *pFlagBrake);
/**
 * @~japanese
 * 位置座標の設定 整数型
 * @param[in] portID 論理ポートID
 * @param[in] pos 絶対位置 単位[Step]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * sets absolute position (integer type)
 * @param[in] portID logical port ID
 * @param[in] pos absolute position. unit[Step]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorPreSetPosition_I32(uint32_t portID,int32_t pos);
/**
 * @~japanese
 * 位置座標の設定 浮動小数点型
 * @param[in] portID 論理ポートID
 * @param[in] pos 絶対位置 単位[radian] または [m]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * sets absolute position (floating point type)
 * @param[in] portID logical port ID
 * @param[in] pos absolute position. unit[radian] or [m]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorPreSetPosition_F64(uint32_t portID,double pos);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * 位置情報の取得
 * @param[in] portID 論理ポートID
 * @param[in] idx 0:現在の検出位置 / 1:指定された目標位置 / 2:現在の指令位置
 * @param[out] *pOutPosition 絶対位置 単位[Step]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * gets absolute position
 * @param[in] portID logical port ID
 * @param[in] idx 0:position sensor / 1:target positione / 2:commanded by profiler
 * @param[out] *pOutPosition. unit[Step]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorGetPosition_I32(uint32_t portID,uint32_t idx,int32_t *pPosition);
/**
 * @~japanese
 * 位置情報の取得
 * @param[in] portID 論理ポートID
 * @param[in] idx 0:現在の検出位置 / 1:指定された目標位置 / 2:現在の指令位置
 * @param[out] *pOutPosition 絶対位置 単位[radian] or [m]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * gets absolute position
 * @param[in] portID logical port ID
 * @param[in] idx 0:position sensor / 1:target positione / 2:commanded by profiler
 * @param[out] *pOutPosition. positione value. unit[radian] or [m]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorGetPosition_F64(uint32_t portID,uint32_t idx,double  *pPosition);
/*---------------------------------------------------------------------------*/
/**
 * @~japanese
 * モーター速度の取得
 * @param[in] portID 論理ポートID
 * @param[in] idx 0:現在の検出速度 / 1:指定された目標速度 / 2:現在の指令速度
 * @param[in] *pOutSpeed 速度の値 単位[Hz]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * gets motor speed
 * @param[in] portID logical port ID
 * @param[in] idx 0:speed sensor / 1:target speed / 2:commanded by profiler
 * @param[in] *pOutSpeed speed value. unit[Hz]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorGetSpeed_I32(uint32_t portID,int32_t idx,int32_t *pOutSpeed);
/**
 * @~japanese
 * モーター速度の取得
 * @param[in] portID 論理ポートID
 * @param[in] idx 0:現在の検出速度 / 1:指定された目標速度 / 2:現在の指令速度
 * @param[in] *pOutSpeed 速度の値 単位[radian/s] or [m/s]
 * @return エラーコード 0:正常 / 0以外:エラーコード
 *
 * @~english
 * gets motor speed
 * @param[in] portID logical port ID
 * @param[in] idx 0:speed sensor / 1:target speed / 2:commanded by profiler
 * @param[in] *pOutSpeed speed value. unit[radian/s] or [m/s]
 * @return the error Code  0:OK / Not 0: error message code
 */
int32_t elMotorGetSpeed_F64(uint32_t portID,int32_t idx,double  *pOutSpeed);
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Ver.2.00 */
/*---------------------------------------------------------------------------*/
/** OpenEL Motor go to HOME
 * @param[in] portID  : ポートID
 * @param[in] idx : 原点検出方法 0:原点センサーを検出するまで進んで停止する
 * @param[in] speed 速度
 * @retval エラーコード EL_OK:正常 / EL_OK以外:異常(運転を受け付けられない状態)
 */
int32_t elMotorGoHome_I32(uint32_t portID,uint32_t idx,int32_t speed);
int32_t elMotorGoHome_F64(uint32_t portID,uint32_t idx,double speed);
/*---------------------------------------------------------------------------*/
/** OpenEL Motor callback function setup
 * @param[in] portID : port ID
 * @param[in] eventID : 0:arriving the position , 1:through the position , 2:arriving the velocity
 * @param[in] pFnc : callback function pointer
 * @param[in] val : position or velocity
 */
int32_t elMotorSetCallBack_I32(uint32_t portID,uint32_t eventID,ELMOTOR_P_FNC_CALLBACK pFncCB,int32_t val);
int32_t elMotorSetCallBack_F64(uint32_t portID,uint32_t eventID,ELMOTOR_P_FNC_CALLBACK pFncCB,double val);
/*---------------------------------------------------------------------------*/

//i 後でopenEL.hに移動する
/*---------------------------------------------------------------------------*/
/** write  parameter
@param[in]	portID  : port ID
@param[in]	prmID  : parameter ID
@param[in]	*pInVal : Value
@return		error code  0:OK / not 0:error (can not execute)
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
int32_t elMotorSetParam_U16(uint32_t portID,uint32_t prmID,uint16_t val);
int32_t elMotorSetParam_I16(uint32_t portID,uint32_t prmID,int16_t  val);
int32_t elMotorSetParam_U32(uint32_t portID,uint32_t prmID,uint32_t val);
int32_t elMotorSetParam_I32(uint32_t portID,uint32_t prmID,uint32_t val);
int32_t elMotorSetParam_I64(uint32_t portID,uint32_t prmID,uint64_t val);
int32_t elMotorSetParam_U64(uint32_t portID,uint32_t prmID,uint64_t val);
int32_t elMotorSetParam_F32(uint32_t portID,uint32_t prmID,float    val);
int32_t elMotorSetParam_F64(uint32_t portID,uint32_t prmID,double   val);
/*---------------------------------------------------------------------------*/
/** read parameter
@param[in]	portID  : port ID
@param[in]	prmID  : parameter ID
@param[in]	*pInVal : Value
@return		error code  0:OK / not 0:error (can not execute)
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/
int32_t elMotorGetParam_U16(uint32_t portID,uint32_t prmID,uint16_t *pOutVal);
int32_t elMotorGetParam_I16(uint32_t portID,uint32_t prmID,int16_t  *pOutVal);
int32_t elMotorGetParam_U32(uint32_t portID,uint32_t prmID,uint32_t *pOutVal);
int32_t elMotorGetParam_I32(uint32_t portID,uint32_t prmID,int32_t  *pOutVal);
int32_t elMotorGetParam_U64(uint32_t portID,uint32_t prmID,uint64_t *pOutVal);
int32_t elMotorGetParam_I64(uint32_t portID,uint32_t prmID,int64_t  *pOutVal);
int32_t elMotorGetParam_F32(uint32_t portID,uint32_t prmID,float    *pOutVal);
int32_t elMotorGetParam_F64(uint32_t portID,uint32_t prmID,double   *pOutVal);
/*---------------------------------------------------------------------------*/
#endif

#endif /* OPENEL_MOTOR_H_ */
